/*
 * (C) Copyright 2002
 * Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 * Alex Zuepke <azu@sysgo.de>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>

#if 0
ulong myflush(void);


#define FLASH_BANK_SIZE	PHYS_FLASH_SIZE
#define MAIN_SECT_SIZE  0x10000		/* 64 KB */

flash_info_t    flash_info[CFG_MAX_FLASH_BANKS];


#define CMD_READ_ARRAY		0x000000F0
#define CMD_UNLOCK1		0x000000AA
#define CMD_UNLOCK2		0x00000055
#define CMD_ERASE_SETUP		0x00000080
#define CMD_ERASE_CONFIRM	0x00000030
#define CMD_PROGRAM		0x000000A0
#define CMD_UNLOCK_BYPASS	0x00000020

#define MEM_FLASH_ADDR1		(*(volatile u16 *)(CFG_FLASH_BASE + (0x00000555 << 1)))
#define MEM_FLASH_ADDR2		(*(volatile u16 *)(CFG_FLASH_BASE + (0x000002AA << 1)))

#define BIT_ERASE_DONE		0x00000080
#define BIT_RDY_MASK		0x00000080
#define BIT_PROGRAM_ERROR	0x00000020
#define BIT_TIMEOUT		0x80000000 /* our flag */

#define READY 1
#define ERR   2
#define TMO   4

/*-----------------------------------------------------------------------
 */

ulong flash_init(void)
{
    int i, j;
    ulong size = 0;

    for (i = 0; i < CFG_MAX_FLASH_BANKS; i++)
    {
	ulong flashbase = 0;
	flash_info[i].flash_id =
#if defined(CONFIG_AMD_LV400)
	  (AMD_MANUFACT & FLASH_VENDMASK) |
	  (AMD_ID_LV400B & FLASH_TYPEMASK);
#elif defined(CONFIG_AMD_LV800)
	  (AMD_MANUFACT & FLASH_VENDMASK) |
	  (AMD_ID_LV800B & FLASH_TYPEMASK);
#else
#error "Unknown flash configured"
#endif
	flash_info[i].size = FLASH_BANK_SIZE;
	flash_info[i].sector_count = CFG_MAX_FLASH_SECT;
	memset(flash_info[i].protect, 0, CFG_MAX_FLASH_SECT);
	if (i == 0)
	  flashbase = PHYS_FLASH_1;
	else
	  panic("configured too many flash banks!\n");
	for (j = 0; j < flash_info[i].sector_count; j++)
	{
	    if (j <= 3)
	    {
		/* 1st one is 16 KB */
		if (j == 0)
		{
			flash_info[i].start[j] = flashbase + 0;
		}

		/* 2nd and 3rd are both 8 KB */
		if ((j == 1) || (j == 2))
		{
			flash_info[i].start[j] = flashbase + 0x4000 + (j-1)*0x2000;
		}

		/* 4th 32 KB */
		if (j == 3)
		{
			flash_info[i].start[j] = flashbase + 0x8000;
		}
	    }
	    else
	    {
		flash_info[i].start[j] = flashbase + (j - 3)*MAIN_SECT_SIZE;
	    }
	}
	size += flash_info[i].size;
    }

    flash_protect(FLAG_PROTECT_SET,
		  CFG_FLASH_BASE,
		  CFG_FLASH_BASE + monitor_flash_len - 1,
		  &flash_info[0]);

    flash_protect(FLAG_PROTECT_SET,
		  CFG_ENV_ADDR,
		  CFG_ENV_ADDR + CFG_ENV_SIZE - 1,
		  &flash_info[0]);

    return size;
}

/*-----------------------------------------------------------------------
 */
void flash_print_info  (flash_info_t *info)
{
    int i;

    switch (info->flash_id & FLASH_VENDMASK)
    {
    case (AMD_MANUFACT & FLASH_VENDMASK):
	printf("AMD: ");
	break;
    default:
	printf("Unknown Vendor ");
	break;
    }

    switch (info->flash_id & FLASH_TYPEMASK)
    {
    case (AMD_ID_LV400B & FLASH_TYPEMASK):
	printf("1x Amd29LV400BB (4Mbit)\n");
	break;
    case (AMD_ID_LV800B & FLASH_TYPEMASK):
	printf("1x Amd29LV800BB (8Mbit)\n");
	break;
    default:
	printf("Unknown Chip Type\n");
	goto Done;
	break;
    }

    printf("  Size: %ld MB in %d Sectors\n",
	   info->size >> 20, info->sector_count);

    printf("  Sector Start Addresses:");
    for (i = 0; i < info->sector_count; i++)
    {
	if ((i % 5) == 0)
	{
	    printf ("\n   ");
	}
	printf (" %08lX%s", info->start[i],
		info->protect[i] ? " (RO)" : "     ");
    }
    printf ("\n");

Done:
}

/*-----------------------------------------------------------------------
 */

int	flash_erase (flash_info_t *info, int s_first, int s_last)
{
    ushort result;
    int iflag, cflag, prot, sect;
    int rc = ERR_OK;
    int chip;

    /* first look for protection bits */

    if (info->flash_id == FLASH_UNKNOWN)
	return ERR_UNKNOWN_FLASH_TYPE;

    if ((s_first < 0) || (s_first > s_last)) {
	return ERR_INVAL;
    }

    if ((info->flash_id & FLASH_VENDMASK) !=
	(AMD_MANUFACT & FLASH_VENDMASK)) {
	return ERR_UNKNOWN_FLASH_VENDOR;
    }

    prot = 0;
    for (sect=s_first; sect<=s_last; ++sect) {
	if (info->protect[sect]) {
	    prot++;
	}
    }
    if (prot)
	return ERR_PROTECTED;

    /*
     * Disable interrupts which might cause a timeout
     * here. Remember that our exception vectors are
     * at address 0 in the flash, and we don't want a
     * (ticker) exception to happen while the flash
     * chip is in programming mode.
     */
    cflag = icache_status();
    icache_disable();
    iflag = disable_interrupts();

    /* Start erase on unprotected sectors */
    for (sect = s_first; sect<=s_last && !ctrlc(); sect++)
    {
	printf("Erasing sector %2d ... ", sect);

	/* arm simple, non interrupt dependent timer */
	reset_timer_masked();

	if (info->protect[sect] == 0)
	{	/* not protected */
	    vu_short *addr = (vu_short *)(info->start[sect]);

	    MEM_FLASH_ADDR1 = CMD_UNLOCK1;
	    MEM_FLASH_ADDR2 = CMD_UNLOCK2;
	    MEM_FLASH_ADDR1 = CMD_ERASE_SETUP;

	    MEM_FLASH_ADDR1 = CMD_UNLOCK1;
	    MEM_FLASH_ADDR2 = CMD_UNLOCK2;
	    *addr = CMD_ERASE_CONFIRM;

	    /* wait until flash is ready */
	    chip = 0;

	    do
	    {
		result = *addr;

		/* check timeout */
		if (get_timer_masked() > CFG_FLASH_ERASE_TOUT)
		{
		    MEM_FLASH_ADDR1 = CMD_READ_ARRAY;
		    chip = TMO;
		    break;
		}

		if (!chip && (result & 0xFFFF) & BIT_ERASE_DONE)
			chip = READY;

		if (!chip && (result & 0xFFFF) & BIT_PROGRAM_ERROR)
			chip = ERR;

	    }  while (!chip);

	    MEM_FLASH_ADDR1 = CMD_READ_ARRAY;

	    if (chip == ERR)
	    {
		rc = ERR_PROG_ERROR;
		goto outahere;
	    }
	    if (chip == TMO)
	    {
		rc = ERR_TIMOUT;
		goto outahere;
	    }

	    printf("ok.\n");
	}
	else /* it was protected */
	{
	    printf("protected!\n");
	}
    }

    if (ctrlc())
      printf("User Interrupt!\n");

outahere:
    /* allow flash to settle - wait 10 ms */
    udelay_masked(10000);

    if (iflag)
      enable_interrupts();

    if (cflag)
      icache_enable();

    return rc;
}

/*-----------------------------------------------------------------------
 * Copy memory to flash
 */

volatile static int write_hword (flash_info_t *info, ulong dest, ushort data)
{
    vu_short *addr = (vu_short *)dest;
    ushort result;
    int rc = ERR_OK;
    int cflag, iflag;
    int chip;

    /*
     * Check if Flash is (sufficiently) erased
     */
    result = *addr;
    if ((result & data) != data)
	return ERR_NOT_ERASED;


    /*
     * Disable interrupts which might cause a timeout
     * here. Remember that our exception vectors are
     * at address 0 in the flash, and we don't want a
     * (ticker) exception to happen while the flash
     * chip is in programming mode.
     */
    cflag = icache_status();
    icache_disable();
    iflag = disable_interrupts();

    MEM_FLASH_ADDR1 = CMD_UNLOCK1;
    MEM_FLASH_ADDR2 = CMD_UNLOCK2;
    MEM_FLASH_ADDR1 = CMD_UNLOCK_BYPASS;
    *addr = CMD_PROGRAM;
    *addr = data;

    /* arm simple, non interrupt dependent timer */
    reset_timer_masked();

    /* wait until flash is ready */
    chip = 0;
    do
    {
	result = *addr;

	/* check timeout */
	if (get_timer_masked() > CFG_FLASH_ERASE_TOUT)
	{
	    chip = ERR | TMO;
	    break;
	}
	if (!chip && ((result & 0x80) == (data & 0x80)))
		chip = READY;

	if (!chip && ((result & 0xFFFF) & BIT_PROGRAM_ERROR))
	{
		result = *addr;

		if ((result & 0x80) == (data & 0x80))
			chip = READY;
		else
			chip = ERR;
	}

    }  while (!chip);

    *addr = CMD_READ_ARRAY;

    if (chip == ERR || *addr != data)
	rc = ERR_PROG_ERROR;

    if (iflag)
      enable_interrupts();

    if (cflag)
      icache_enable();

    return rc;
}

/*-----------------------------------------------------------------------
 * Copy memory to flash.
 */

int write_buff (flash_info_t *info, uchar *src, ulong addr, ulong cnt)
{
    ulong cp, wp;
    int l;
    int i, rc;
    ushort data;

    wp = (addr & ~1);	/* get lower word aligned address */

    /*
     * handle unaligned start bytes
     */
    if ((l = addr - wp) != 0) {
	data = 0;
	for (i=0, cp=wp; i<l; ++i, ++cp) {
	    data = (data >> 8) | (*(uchar *)cp << 8);
	}
	for (; i<2 && cnt>0; ++i) {
	    data = (data >> 8) | (*src++ << 8);
	    --cnt;
	    ++cp;
	}
	for (; cnt==0 && i<2; ++i, ++cp) {
	    data = (data >> 8) | (*(uchar *)cp << 8);
	}

	if ((rc = write_hword(info, wp, data)) != 0) {
	    return (rc);
	}
	wp += 2;
    }

    /*
     * handle word aligned part
     */
    while (cnt >= 2) {
	data = *((vu_short*)src);
	if ((rc = write_hword(info, wp, data)) != 0) {
	    return (rc);
	}
	src += 2;
	wp  += 2;
	cnt -= 2;
    }

    if (cnt == 0) {
	return ERR_OK;
    }

    /*
     * handle unaligned tail bytes
     */
    data = 0;
    for (i=0, cp=wp; i<2 && cnt>0; ++i, ++cp) {
	data = (data >> 8) | (*src++ << 8);
	--cnt;
    }
    for (; i<2; ++i, ++cp) {
	data = (data >> 8) | (*(uchar *)cp << 8);
    }

    return write_hword(info, wp, data);
}

#endif

#if defined(CFG_ENV_IS_IN_FLASH)
# ifndef  CFG_ENV_ADDR
#  define CFG_ENV_ADDR	(CFG_FLASH_BASE + CFG_ENV_OFFSET)
# endif
# ifndef  CFG_ENV_SIZE
#  define CFG_ENV_SIZE	CFG_ENV_SECT_SIZE
# endif
# ifndef  CFG_ENV_SECT_SIZE
#  define CFG_ENV_SECT_SIZE  CFG_ENV_SIZE
# endif
#endif

#define	FLASH_ID_MASK			0xFFFFFFFF
#define FLASH_BLOCK_SIZE		0x00040000
#define FLASH_CMD_READ_ID		0x00900090
#define FLASH_CMD_RESET			0x00ff00ff
#define FLASH_CMD_BLOCK_ERASE		0x00200020
#define FLASH_CMD_ERASE_CONFIRM		0x00D000D0
#define FLASH_CMD_CLEAR_STATUS		0x00500050
#define FLASH_CMD_SUSPEND_ERASE		0x00B000B0
#define FLASH_CMD_WRITE			0x00400040
#define FLASH_CMD_PROTECT		0x00600060
#define FLASH_CMD_PROTECT_SET		0x00010001
#define FLASH_CMD_PROTECT_CLEAR		0x00D000D0
#define FLASH_STATUS_DONE		0x00800080

flash_info_t	flash_info[CFG_MAX_FLASH_BANKS];

/*
 * Local function prototypes
 */
static ulong 	flash_get_size 		(vu_short *addr, flash_info_t *info);
int 	write_buff 		(flash_info_t *info, ulong dest, ushort data);
static void 	flash_get_offsets 	(ulong base, flash_info_t *info);

/*
 * Initialize flash
 */

unsigned long flash_init (void)
{
	unsigned long size_b0;
	int i;

	/* Init: no FLASHes known */
	for (i=0; i<CFG_MAX_FLASH_BANKS; ++i) {
		flash_info[i].flash_id = FLASH_UNKNOWN;
	}

	/* Static FLASH Bank configuration here - FIXME XXX */
#if 1
	debug ("\n## Get flash bank 1 size @ 0x%08x\n",CFG_FLASH_BASE);
#endif
	size_b0 = flash_get_size((vu_long *)CFG_FLASH_BASE, &flash_info[0]);

	if (flash_info[0].flash_id == FLASH_UNKNOWN) {
		printf ("## Unknown FLASH on Bank 0: "
			"ID 0x%lx, Size = 0x%08lx = %ld MB\n",
			flash_info[0].flash_id,
			size_b0, size_b0<<20);
	}

	flash_get_offsets (CFG_FLASH_BASE, &flash_info[0]);

	flash_info[0].size = size_b0;

#if CFG_MONITOR_BASE >= CFG_FLASH_BASE
	/* monitor protection ON by default */
	flash_protect(FLAG_PROTECT_SET,
		      CFG_MONITOR_BASE,
		      CFG_MONITOR_BASE+monitor_flash_len-1,
		      &flash_info[0]);
#endif

#ifdef	CFG_ENV_IS_IN_FLASH
	/* ENV protection ON by default */
	flash_protect(FLAG_PROTECT_SET,
		      CFG_ENV_ADDR,
		      CFG_ENV_ADDR+CFG_ENV_SECT_SIZE-1,
		      &flash_info[0]);
#endif

	return size_b0;
}

/*
 * Compute start adress of each sector (block)
 */

static void flash_get_offsets (ulong base, flash_info_t *info)
{
	int i;

	if (info->flash_id == FLASH_UNKNOWN) {
		return;
	}

	switch (info->flash_id & FLASH_VENDMASK) {
	case FLASH_MAN_INTEL:
	    for (i = 0; i < info->sector_count; i++) {
		info->start[i] = base + i * FLASH_BLOCK_SIZE;
	    }
	    return;

	default:
	    printf ("Don't know sector offsets for flash type 0x%lx\n",
		info->flash_id);
	    return;
	}
}

/*
 * Print flash information
 */
void flash_print_info  (flash_info_t *info)
{
	int i;

	if (info->flash_id == FLASH_UNKNOWN) {
		printf ("missing or unknown FLASH type\n");
		return;
	}

	switch (info->flash_id & FLASH_VENDMASK) {
	case FLASH_MAN_AMD:	printf ("AMD ");		break;
	case FLASH_MAN_FUJ:	printf ("Fujitsu ");		break;
	case FLASH_MAN_SST:	printf ("SST ");		break;
	case FLASH_MAN_STM:	printf ("STM ");		break;
	case FLASH_MAN_INTEL:	printf ("Intel ");		break;
	case FLASH_MAN_MT:	printf ("MT ");			break;
	default:		printf ("Unknown Vendor ");	break;
	}

	switch (info->flash_id & FLASH_TYPEMASK) {
	case FLASH_28F320J3A:	printf ("28F320J3A (32Mbit) 16-Bit\n");
				break;
	case FLASH_28F640J3A:	printf ("28F640J3A (64Mbit) 16-Bit\n");
				break;
	case FLASH_28F128J3A:	printf ("28F128J3A (128Mbit) 16-Bit\n");
				break;
	default:		printf ("Unknown Chip Type\n");
				break;
	}

	if (info->size >= (1 << 20)) {
		i = 20;
	} else {
		i = 10;
	}
	printf ("  Size: %ld %cB in %d Sectors\n",
		info->size >> i,
		(i == 20) ? 'M' : 'k',
		info->sector_count);

	printf ("  Sector Start Addresses:");
	for (i=0; i<info->sector_count; ++i) {
		if ((i % 5) == 0)
			printf ("\n   ");
		printf (" %08lX%s",
			info->start[i],
			info->protect[i] ? " (RO)" : "     "
		);
	}
	printf ("\n");
	return;
}

/*
 * Get size of flash in bytes.
 * The following code cannot be run from FLASH!
 */

static ulong flash_get_size (vu_long *addr, flash_info_t *info)
{
	vu_long value;

	/* Read Manufacturer ID */
	addr[0] = FLASH_CMD_READ_ID;
	value = addr[0];

	switch (value) {
	case (AMD_MANUFACT & FLASH_ID_MASK):
		info->flash_id = FLASH_MAN_AMD;
		break;
	case (FUJ_MANUFACT & FLASH_ID_MASK):
		info->flash_id = FLASH_MAN_FUJ;
		break;
	case (SST_MANUFACT & FLASH_ID_MASK):
		info->flash_id = FLASH_MAN_SST;
		break;
	case (STM_MANUFACT & FLASH_ID_MASK):
		info->flash_id = FLASH_MAN_STM;
		break;
	case (INTEL_MANUFACT & FLASH_ID_MASK):
		info->flash_id = FLASH_MAN_INTEL;
		break;
	default:
		info->flash_id = FLASH_UNKNOWN;
		info->sector_count = 0;
		info->size = 0;
		addr[0] = FLASH_CMD_RESET;	/* restore read mode */
		return (0);			/* no or unknown flash	*/
	}

	value = addr[1];			/* device ID		*/

	switch (value) {
	case (INTEL_ID_28F320J3A  & FLASH_ID_MASK):
		info->flash_id += FLASH_28F320J3A;
		info->sector_count = 32;
		info->size = 0x00800000;
		break;				/* =>  32 MBit * 2		*/

	case (INTEL_ID_28F640J3A & FLASH_ID_MASK):
		info->flash_id += FLASH_28F640J3A;
		info->sector_count = 64;
		info->size = 0x01000000;
		break;				/* => 64 MBit * 2		*/

	case (INTEL_ID_28F128J3A & FLASH_ID_MASK):
		info->flash_id += FLASH_28F128J3A;
		info->sector_count = 128;
		info->size = 0x02000000;
		break;				/* => 128 MBit * 2		*/

	default:
		info->flash_id = FLASH_UNKNOWN;
		addr[0] = FLASH_CMD_RESET;	/* restore read mode */
		return (0);			/* => no or unknown flash */

	}

	if (info->sector_count > CFG_MAX_FLASH_SECT) {
		printf ("** ERROR: sector count %d > max (%d) **\n",
			info->sector_count, CFG_MAX_FLASH_SECT);
		info->sector_count = CFG_MAX_FLASH_SECT;
	}

	addr[0] = FLASH_CMD_RESET;		/* restore read mode */

	return (info->size);
}


/*
 * Erase unprotected sectors
 */

int flash_erase (flash_info_t *info, int s_first, int s_last)
{
	int flag, prot, sect;
	ulong start, now, last;

	if ((s_first < 0) || (s_first > s_last)) {
		if (info->flash_id == FLASH_UNKNOWN) {
			printf ("- missing\n");
		} else {
			printf ("- no sectors to erase\n");
		}
		return 1;
	}

	if ((info->flash_id & FLASH_VENDMASK) != FLASH_MAN_INTEL) {
		printf ("Can erase only Intel flash types - aborted\n");
		return 1;
	}

	prot = 0;
	for (sect=s_first; sect<=s_last; ++sect) {
		if (info->protect[sect]) {
			prot++;
		}
	}

	if (prot) {
		printf ("- Warning: %d protected sectors will not be erased!\n",
			prot);
	} else {
		printf ("\n");
	}

	start = get_timer (0);
	last  = start;

	/* Start erase on unprotected sectors */
	for (sect = s_first; sect<=s_last; sect++) {
		if (info->protect[sect] == 0) {	/* not protected */
			vu_long *addr = (vu_long *)(info->start[sect]);
			unsigned long status;

			/* Disable interrupts which might cause a timeout here */
			flag = disable_interrupts();

#ifdef DEBUG
			printf("Erase sector %d at start addr 0x%08X", sect, (unsigned int)info->start[sect]);
#endif

			*addr = FLASH_CMD_CLEAR_STATUS;
			*addr = FLASH_CMD_BLOCK_ERASE;
			*addr = FLASH_CMD_ERASE_CONFIRM;

			/* re-enable interrupts if necessary */
			if (flag)
				enable_interrupts();

			/* wait at least 80us - let's wait 1 ms */
			udelay (1000);

			while (((status = *addr) & FLASH_STATUS_DONE) != FLASH_STATUS_DONE) {
				if ((now=get_timer(start)) > CFG_FLASH_ERASE_TOUT) {
					printf("Flash erase timeout at address %lx\n", info->start[sect]);
					*addr = FLASH_CMD_SUSPEND_ERASE;
					*addr = FLASH_CMD_RESET;
					return 1;
				}

				/* show that we're waiting */
				if ((now - last) > 1000) {	/* every second */
					putc ('.');
					last = now;
				}
			}
			*addr = FLASH_CMD_RESET;
		}
	}
	printf (" done\n");
	return 0;
}

/*-----------------------------------------------------------------------
 * Copy memory to flash
 */

static int write_dword (flash_info_t * info, ulong dest, ulong data)
{
	vu_long *addr = (vu_long *) dest, val;
	int rc = ERR_OK;
	int flag;

	/* Check if Flash is (sufficiently) erased
	 */
	if ((*addr & data) != data)
		return ERR_NOT_ERASED;

	/*
	 * Disable interrupts which might cause a timeout
	 * here. Remember that our exception vectors are
	 * at address 0 in the flash, and we don't want a
	 * (ticker) exception to happen while the flash
	 * chip is in programming mode.
	 */
	flag = disable_interrupts ();

	/* clear status register command */
	*addr = 0x00500050;

	/* program set-up command */
	*addr = 0x00400040;

	/* latch address/data */
	*addr = data;

	/* arm simple, non interrupt dependent timer */
	reset_timer_masked ();

	/* wait while polling the status register */
	while (((val = *addr) & 0x00800080) != 0x00800080) {
		if (get_timer_masked () > CFG_FLASH_WRITE_TOUT) {
			rc = ERR_TIMOUT;
			/* suspend program command */
			*addr = 0x00B000B0;
			goto outahere;
		}
	}

	if (val & 0x001A001A) {			/* check for error */
		printf ("\nFlash write error %02x at address %08lx\n",
				(unsigned long) val, (unsigned long) dest);
		if (val & ((1 << 3)|(1 << 19))) {
			printf ("Voltage range error.\n");
			rc = ERR_PROG_ERROR;
			goto outahere;
		}
		if (val & ((1 << 1)|(1 << 17))) {
			printf ("Device protect error.\n");
			rc = ERR_PROTECTED;
			goto outahere;
		}
		if (val & ((1 << 4)||(1 << 20))) {
			printf ("Programming error.\n");
			rc = ERR_PROG_ERROR;
			goto outahere;
		}
		rc = ERR_PROG_ERROR;
		goto outahere;
	}

  outahere:
	/* read array command */
	*addr = 0x00FF00FF;

	if (flag)
		enable_interrupts ();

	return rc;
}

/*-----------------------------------------------------------------------
 * Copy memory to flash.
 */

int write_buff (flash_info_t * info, uchar * src, ulong addr, ulong cnt)
{
	ulong cp, wp;
	ulong data;
	int l;
	int i, rc;

    if(cnt==0) 
        return 0;

	wp = (addr & ~3);			/* get lower word aligned address */

	/*
	 * handle unaligned start bytes
	 */

    /*
	if ((l = addr - wp) != 0) {
		data = 0;
		for (i = 0, cp = wp; i < l; ++i, ++cp) {
			data = (data >> 8) | (*(uchar *) cp << 8);
		}
		for (; i < 2 && cnt > 0; ++i) {
			data = (data >> 8) | (*src++ << 8);
			--cnt;
			++cp;
		}
		for (; cnt == 0 && i < 2; ++i, ++cp) {
			data = (data >> 8) | (*(uchar *) cp << 8);
		}

		if ((rc = write_word (info, wp, data)) != 0) {
			return (rc);
		}
		wp += 2;
	}
    */

    if ((l = addr - wp) != 0) {

        cp = wp;
        switch(l) {
            case 1:
                if(cnt==1){
                    data = (*(uchar*)cp)|(*src << 8)|(*(uchar*)(cp+2) << 16)|(*(uchar*)(cp+3) << 24);
                    if ((rc = write_dword (info, wp, data)) != 0) {
			            return (rc);
		            }
		            return 0;
                }
                else if(cnt==2){
                    data = (*(uchar*)cp)|(*src << 8)|(*(src+1) << 16)|(*(uchar*)(cp+3) << 24);
                    if ((rc = write_dword (info, wp, data)) != 0) {
			            return (rc);
		            }
		            return 0;
                }
                else if(cnt==3){
                    data = (*(uchar*)cp)|(*src << 8)|(*(src+1) << 16)|(*(src+2) << 24);
                    if ((rc = write_dword (info, wp, data)) != 0) {
			            return (rc);
		            }
		            return 0;
                }
                else {
                    data = (*(uchar*)cp)|(*src << 8)|(*(src+1) << 16)|(*(src+2) << 24);
                    if ((rc = write_dword (info, wp, data)) != 0) {
			            return (rc);
		            }
                    cnt = cnt - 3;
                    src += 3;
                    wp += 4;
                }
                break;
            case 2:
                if(cnt==1){
                    data = (*(uchar*)cp)|(*(uchar*)(cp+1) << 8)|(*src << 16)|(*(uchar*)(cp+3) << 24);
                    if ((rc = write_dword (info, wp, data)) != 0) {
			            return (rc);
		            }
		            return 0;
                }
                else if(cnt==2){
                    data = (*(uchar*)cp)|(*(uchar*)(cp+1) << 8)|(*(src+1) << 16)|(*(src+2) << 24);
                    if ((rc = write_dword (info, wp, data)) != 0) {
			            return (rc);
		            }
		            return 0;
                }
                else {
                    data = (*(uchar*)cp)|(*(uchar*)(cp+1) << 8)|(*src << 16)|(*(src+1) << 24);
                    if ((rc = write_dword (info, wp, data)) != 0) {
			            return (rc);
		            }
                    cnt = cnt - 2;
                    src += 2;
		            wp += 4;
                }
                break;
            case 3:
                if(cnt==1){
                    data = (*(uchar*)cp)|(*(uchar*)(cp+1) << 8)|(*(uchar*)(cp+2) << 16)|(*(src+2) << 24);
                    if ((rc = write_dword (info, wp, data)) != 0) {
			            return (rc);
		            }
		            return 0;
                }
                else {
                    data = (*(uchar*)cp)|(*(uchar*)(cp+1) << 8)|(*(uchar*)(cp+2) << 16)|(*(src+2) << 24);
                    if ((rc = write_dword (info, wp, data)) != 0) {
			            return (rc);
		            }
                    cnt = cnt - 1;
                    src += 1;
		            wp += 4;
                }
                break;
            default:
                break;                
        }

    }
    
	/*
	 * handle word aligned part
	 */
	while (cnt >= 4) {
		data = *((vu_long *) src);
		if ((rc = write_dword (info, wp, data)) != 0) {
			return (rc);
		}
		src += 4;
		wp += 4;
		cnt -= 4;
	}

	if (cnt == 0) {
		return ERR_OK;
	}

	/*
	 * handle unaligned tail bytes
	 */

    /*
	data = 0;
	for (i = 0, cp = wp; i < 2 && cnt > 0; ++i, ++cp) {
		data = (data >> 8) | (*src++ << 8);
		--cnt;
	}
	for (; i < 2; ++i, ++cp) {
		data = (data >> 8) | (*(uchar *) cp << 8);
	}
	*/

    cp = wp;
    
	switch(cnt) {
        case 1:
            data = (*src)|(*(uchar*)(cp+1) << 8)|(*(uchar*)(cp+2) << 16)|(*(uchar*)(cp+3) << 24);
        case 2:
            data = (*src)|(*(src+1) << 8)|(*(uchar*)(cp+2) << 16)|(*(uchar*)(cp+3) << 24);
        case 3:
            data = (*src)|(*(src+1) << 8)|(*(src+2) << 16)|(*(uchar*)(cp+3) << 24);
        default:
            return 0;
	}

	return write_dword (info, wp, data);
}


int flash_real_protect(flash_info_t *info, long sector, int prot)
{
	vu_long *addr = (vu_long*)(info->start[sector]);
	ulong start;

	*addr = FLASH_CMD_CLEAR_STATUS;
	*addr = FLASH_CMD_PROTECT;

	if(prot) {
		*addr = FLASH_CMD_PROTECT_SET;
	} else {
		*addr = FLASH_CMD_PROTECT_CLEAR;
	}

	/* wait for error or finish */
	start = get_timer (0);
	while(!(addr[0] & FLASH_STATUS_DONE)){
		if (get_timer(start) > CFG_FLASH_ERASE_TOUT) {
			printf("Flash protect timeout at address %lx\n",  info->start[sector]);
			addr[0] = FLASH_CMD_RESET;
			return (1);
		}
	}
	/* Set software protect flag */
	info->protect[sector] = prot;
	*addr = FLASH_CMD_RESET;
	return (0);
}

